//
//  TodoList.swift
//  Do It
//
//  Created by Jim Dovey on 8/25/19.
//  Copyright © 2019 Jim Dovey. All rights reserved.
//

import SwiftUI
import Combine

struct ErrorInfo: Identifiable {
    let error: Error
    let id: UUID = UUID()
    
    init(_ error: Error) {
        self.error = error
    }
}

fileprivate let newItemCommand = KeyCommand(
    title: NSLocalizedString("New Item", comment: "Key command title"),
    input: "n",
    modifierFlags: [.command],
    discoverabilityTitle: NSLocalizedString("Create a new to-do item",
                                comment: "Key command discoverability title"))
fileprivate let listInfoCommand = KeyCommand(
    title: NSLocalizedString("List Info", comment: "Key command title"),
    input: "i",
    modifierFlags: [.command],
    discoverabilityTitle: NSLocalizedString("Show or edit list properties",
                                comment: "Key command discoverability title"))

struct TodoList: View {
    private enum ListData {
        case list(TodoItemList)
        case group(TodoItemGroup)
    }
    
    @State private var sortBy: SortOption = .manual
    @State private var showingChooser: Bool = false
    @Environment(\.presentationMode) private var presentationMode
    @EnvironmentObject private var data: DataCenter

    private static let itemTemplate = TodoItem(
        title: "New Item", priority: .normal, listID: UUID())
    
    @State private var editingItem = Self.itemTemplate
    @State private var listData: ListData
    
    @Environment(\.errorPublisher) private var errorPublisher
    
    @State private var showItemEditor = false
    @State private var showListEditor = false
    
    @State private var selectedItems: Set<UUID> = []
    @Environment(\.undoManager) var undoManager

    init(list: TodoItemList) {
        self._listData = State(wrappedValue: .list(list))
    }

    init(group: TodoItemGroup) {
        self._listData = State(wrappedValue: .group(group))
    }

    var body: some View {
        List(selection: $selectedItems) {
            // START:OnDrop
            ForEach(sortedItems) { item in
                // <literal:elide> ... </literal:elide>
                // END:OnDrop
                NavigationLink(
                    destination: TodoItemDetail(item: item)
                        .environmentObject(self.data)
                ) {
                    TodoItemRow(item: item)
                        .accentColor(self.color(for: item))
                }
                // START:OnDrop
            }
            // <literal:elide>onDelete</literal:elide>
            // <literal:elide>onMove</literal:elide>
            // END:OnDrop
            .onDelete { self.removeTodoItems(atOffsets: $0) }
            .onMove(perform: self.sortBy == .manual && isList
                ? { self.moveTodoItems(fromOffsets: $0, to: $1) }
                : nil)
            // START:OnDrop
            .onInsert(of: self.droppableUTIs,
                      perform: self.handleDrop(at:providers:))
            // END:OnDrop
        }
        .navigationBarTitle(title)
        .navigationBarItems(trailing: barItems)
        .listStyle(GroupedListStyle())
        .onKeyCommand(newItemCommand) {
            guard !self.showItemEditor && !self.showListEditor else {
                return
            }
            self.editingItem = Self.itemTemplate
            self.editingItem.listID = self.list?.id ?? self.data.defaultListID
            self.showItemEditor.toggle()
        }
        .onKeyCommand(listInfoCommand) {
            guard !self.showItemEditor && !self.showListEditor, let _ = self.list else {
                return
            }
            
            self.showListEditor.toggle()
        }
        .actionSheet(isPresented: $showingChooser) {
            ActionSheet(
                title: Text("Sort Order"),
                buttons: SortOption.allCases.map { opt in
                    ActionSheet.Button.default(Text(opt.title)) {
                        self.sortBy = opt
                    }
            })
        }
        .onReceive(data.$todoItems.combineLatest(data.$todoLists).receive(on: RunLoop.main)) { _ in
            self.updateData()
        }
    }
}

// MARK: - Helper Properties

extension TodoList {
    private var sortButton: some View {
        Button(action: { self.showingChooser.toggle() }) {
            Image(systemName: "arrow.up.arrow.down.circle.fill")
                .imageScale(.large)
                .accessibility(label: Text("Sort List"))
        }
    }

    private var addButton: some View {
        Button(action: {
            self.editingItem = Self.itemTemplate
            self.editingItem.listID = self.list?.id ?? self.data.defaultListID
            self.showItemEditor.toggle()
        }) {
            Image(systemName: "plus.circle.fill")
                .imageScale(.large)
                .accessibility(label: Text("Add New To-Do Item"))
        }
        .accessibility(label: Text("Add a new To-Do Item"))
        .popover(isPresented: $showItemEditor) {
            self.editorSheet
                .environmentObject(self.data)
                .frame(idealWidth: 500, idealHeight: 600)
        }
    }

    private var editorSheet: some View {
        let done = Button(action:{
            self.data.addTodoItem(self.editingItem)
            self.showItemEditor = false
        }) {
            Text("Done")
                .bold()
        }
        let cancel = Button("Cancel") {
            self.showItemEditor = false
        }
        return NavigationView {
            TodoItemEditor(item: $editingItem)
                .navigationBarItems(leading: cancel, trailing: done)
        }
    }

    private var barItems: some View {
        return HStack(spacing: 14) {
            if isList {
                Button(action: { self.showListEditor.toggle() }) {
                    Image(systemName: "info.circle")
                        .imageScale(.large)
                }
                .iconHoverEffect()
                .popover(isPresented: $showListEditor) {
                    TodoListEditor(list: self.list!)
                        .environmentObject(self.data)
                        .frame(idealWidth: 500, idealHeight: 600)
                }
            }
            sortButton.iconHoverEffect()
            addButton.iconHoverEffect()
            EditButton().niceHoverEffect()
        }
    }

    private var isList: Bool {
        if case .list = self.listData {
            return true
        }
        return false
    }
    
    private var list: TodoItemList? {
        if case let .list(list) = self.listData {
            return list
        }
        return nil
    }
    
    private var isAll: Bool {
        if case let .group(g) = self.listData, g == .all {
            return true
        }
        return false
    }
    
    // START:DroppableUTIs
    private var droppableUTIs: [String] {
        switch listData {
        case .list: return [rawTextUTI, todoItemUUIDUTI, todoItemUTI, jsonUTI]
        case .group(.all): return [rawTextUTI, todoItemUTI, jsonUTI]
        default: return []
        }
    }
    // END:DroppableUTIs

    private func forciblyDismiss() {
        presentationMode.wrappedValue.dismiss()
    }

    private var items: [TodoItem] {
        switch listData {
        case .list(let list): return data.items(in: list)
        case .group(let group): return group.items(from: data)
        }
    }

    private var title: LocalizedStringKey {
        switch listData {
        case .list(let list): return LocalizedStringKey(list.name)
        case .group(let group): return group.title
        }
    }

    private func color(for item: TodoItem) -> Color {
        switch listData {
        case .list(let list): return list.color.uiColor
        case .group(let group): return group.color
        }
    }
}

// MARK: - Sorting

extension TodoList {
    private var sortedItems: [TodoItem] {
        if case .manual = sortBy { return items }

        return items.sorted {
            switch sortBy {
            case .title:
                return $0.title.lowercased() < $1.title.lowercased()
            case .priority:
                return $0.priority > $1.priority
            case .dueDate:
                return ($0.date ?? .distantFuture) <
                    ($1.date ?? .distantFuture)
            case .manual:
                fatalError("unreachable")
            }
        }
    }
}

// MARK: - Model Manipulation

extension TodoList {
    private var usingUnchangedList: Bool {
        sortBy == .manual
    }

    private func removeTodoItems(atOffsets offsets: IndexSet) {
        if let list = list {
            data.removeTodoItems(atOffsets: offsets, in: list)
        }
        else {
            let items = sortedItems
            let uuids = offsets.map { items[$0].id }
            data.removeTodoItems(withIDs: uuids)
        }
    }

    private func moveTodoItems(fromOffsets offsets: IndexSet, to newIndex: Int) {
        guard let list = self.list else { return }
        data.moveTodoItems(fromOffsets: offsets, to: newIndex, within: list)
    }

    private func updateData() {
        switch listData {
        case let .list(list):
            if let newList = data.todoLists.first(where: { $0.id == list.id }) {
                listData = .list(newList)
            }
            else {
                // List is gone!
                forciblyDismiss()
            }

        case .group:
            break
        }
    }
    
    private func handleDrop(at index: Int, providers: [NSItemProvider]) {
        let list: TodoItemList
        if case let .list(l) = listData {
            list = l
        }
        else if case let .group(t) = listData, t == .all {
            list = data.defaultItemList
        }
        else {
            return
        }
        
        ItemReceiver(dataCenter: data).readFromProviders(providers) { output, err in
            if let error = err {
                self.errorPublisher.send(error)
                return
            }
            guard !output.isEmpty else {
                self.errorPublisher.send(ItemProviderError.noData)
                return
            }

            // START:HandleExistingItem
            for value in output {
                switch value {
                case .existingItem(let item):
                    if item.listID == list.id {
                        // just move it up or down
                        let items = self.data.items(in: list)
                        if let curIndex = items.firstIndex(where: {$0.id == item.id}) {
                            let indices = IndexSet(integer: curIndex)
                            self.data.moveTodoItems(fromOffsets: indices,
                                                    to: index, within: list)
                            break
                        }
                    }
                    else {
                        self.data.moveTodoItems(withIDs: [item.id],
                                                toList: list,
                                                at: index)
                    }
                    
                    // END:HandleExistingItem
                    // START:HandleNewItem
                case .item(let item):
                    // insert at end of list
                    let items = self.data.items(in: list)
                    var newItem = item
                    newItem.listID = list.id
                    if case .group(.all) = self.listData {
                        self.data.addTodoItem(newItem, globalIndex: index)
                    }
                    else {
                        self.data.undoManager?.beginUndoGrouping()
                        self.data.addTodoItem(newItem)
                        let offsets = IndexSet(integer: items.count)
                        self.data.moveTodoItems(fromOffsets: offsets,
                                                to: index, within: list)
                        self.data.undoManager?.endUndoGrouping()
                    }
                    // END:HandleNewItem
                    
                    // START:HandleString
                case .string(let str):
                    // new item with this as its title
                    let items = self.data.items(in: list)
                    let newItem = TodoItem(title: str, priority: .normal,
                                           listID: list.id)
                    if case .group(.all) = self.listData {
                        self.data.addTodoItem(newItem, globalIndex: index)
                    }
                    else {
                        self.data.undoManager?.beginUndoGrouping()
                        self.data.addTodoItem(newItem)
                        let offsets = IndexSet(integer: items.count)
                        self.data.moveTodoItems(fromOffsets: offsets,
                                                to: index, within: list)
                        self.data.undoManager?.endUndoGrouping()
                    }
                    // END:HandleString
                    
                    // START:HandleExistingItem
                default:
                    self.errorPublisher
                        .send(ItemProviderError.unsupportedDataType)
                    return
                }
            }
            // END:HandleExistingItem
        }
    }
}

fileprivate enum SortOption: String, CaseIterable {
    case title = "Title"
    case priority = "Priority"
    case dueDate = "Due Date"
    case manual = "Manual"

    var title: LocalizedStringKey { LocalizedStringKey(rawValue) }
}

struct TodoList_Previews: PreviewProvider {
    static var previews: some View {
        return NavigationView {
            TodoList(group: .all)
                .environmentObject(DataCenter())
        }
        .previewDevice("iPhone 11 Pro")
    }
}
